# -------------------------------------------------------------------
# Section 2:  ggplot2 Basics
# -------------------------------------------------------------------

library(ggplot2)

data() # see all available base datasets
data(package="ggplot2") # see datasets available via ggplot2
data(diamonds) # load the diamonds dataset

head(diamonds) # see the first few rows
nrow(diamonds) # count the number of rows
summary(diamonds) # get summary statistics
str(diamonds) # show the structure of the dataframe

ggplot(data=diamonds, aes(x=clarity) ) + geom_bar()
ggplot(data=diamonds, aes(x=clarity, fill=color) ) + geom_bar()
ggplot(data=diamonds, aes(x=price) ) + geom_histogram()

g <- ggplot(data=diamonds, aes(x=clarity, y=carat) )
g + geom_point() # scatterplot
g + geom_jitter() # scatterplot with jitter
g + geom_boxplot() # a boxplot

# -------------------------------------------------------------------
# Section 3:  Graphically Exploring the Diamonds Dataset with ggplot2
# -------------------------------------------------------------------

g <- ggplot(data=diamonds, aes(x=carat, y=price) )

# a simple scatterplot
g + geom_point()

# scatterplot with different colors/shapes by group(s)
g + geom_point(aes(color=color))
g + geom_point(aes(color=color, shape=cut))

# scatterplots with only part of the data
g_subset <- ggplot(data=subset(diamonds, color=="J"), aes(x=carat, y=price) )

g_subset + geom_point()
g_subset + geom_point(aes(color=color))

# scatterplot with LOESS fit
g + geom_point() + 
    geom_smooth()

g + geom_point() + 
    geom_smooth(se=FALSE) # no error band

# faceting by cut
g + geom_point(aes(color=color)) + 
    facet_wrap(~cut)

g + geom_point(aes(color=color)) + 
    facet_grid(.~cut)

g + geom_point(aes(color=color)) + 
    facet_grid(clarity~cut)

# -------------------------------------------------------------------
# Section 4:  Creating "new" Plots
# -------------------------------------------------------------------

#--------------------------#
# 4.1: Polar Bars and Pies #
#--------------------------#

# simple polar bar
ggplot(data=diamonds, aes(x=clarity, fill=color) ) + 
  geom_bar() + 
  coord_polar()

# fancy polar bar
ggplot(data=diamonds, aes(x=clarity, fill=color) ) + 
  geom_bar() + 
  coord_polar() + 
  scale_x_discrete(name="") + # no label on x-axis
  scale_y_continuous(name="") + # no label on y-axis
  theme_bw() + # remove the ggplot2 background
  opts(
    axis.ticks=theme_blank(), # no axis ticks
    legend.position="none", # remove legend
    axis.text.x=theme_blank(), # remove r-axis labels
    axis.text.y=theme_blank() # remove theta-axis labels
  )

# piechart
ggplot(data=diamonds, aes(x=factor(1), fill=cut) ) + 
  geom_bar(width=1) + # fatter bars 
  coord_polar(theta="y") # (x,y) -> (r,theta)

#-------------#
# 4.2: opts() #
#-------------#

?opts

#-------------------------------#
# 4.3 Regression Lines by Group #
#-------------------------------#

model <- lm(price ~ carat + color, data=diamonds) 

grid <- with(diamonds, 
  base::expand.grid(
    carat = seq(min(carat), max(carat), length = 20), 
    color = levels(factor(color)) 
  )
)

grid$price <- stats::predict(model, newdata=grid)

g <- ggplot(data=diamonds, aes(x=carat, y=price, color=color) ) + 
  geom_point() +  
  geom_line(data=grid) + 
  ylim(0,22000)

g

# faceted
g + facet_wrap(~color)

# Adding confidence bands
err <- stats::predict(model, newdata=grid, se = TRUE) 
grid$ucl <- err$fit + 1.96 * err$se.fit 
grid$lcl <- err$fit - 1.96 * err$se.fit 

g + facet_wrap(~color) +
  geom_smooth(aes(ymin = lcl, ymax = ucl), data=grid, stat="identity")

#--------------#
# 4.4 Heatmaps #
#--------------#

g <- ggplot(diamonds, aes(cut, color)) +
  geom_tile(aes(fill=depth), color="white")

g

g + theme_bw() + 
  scale_fill_gradient(low="#2E2EFE", high="#FA5858") +
  scale_x_discrete(expand=c(0,0)) + 
  scale_y_discrete(expand=c(0,0)) + 
  labs(x="", y="") +
  opts(legend.position="none", 
       axis.ticks=theme_blank(), 
       axis.text.x=theme_text(size = 9, 
                   angle = 320, 
                   hjust = 0, 
                   colour = "grey10"
       ),
       title="Depth of Diamond"
  )

#-------------------------------------#
# 4.5 Multiple plots without faceting #
#-------------------------------------#

# Setting up the plots
g1 <- ggplot(data=diamonds, aes(x=clarity, y=price) )
g2 <- ggplot(data=diamonds, aes(x=price))


first <- g1 + 
         geom_boxplot() + 
         opts(title="Boxplots") + 
         scale_y_log10("Log-scale Price") +
         scale_x_discrete("Clarity") 

second <- g1 +
          geom_jitter(size=1, aes(color=color)) +
          opts(title="Jitter Plot") + 
          scale_x_discrete("Clarity") +
          scale_y_continuous("Price")

third <- g2 + 
         geom_histogram(aes(y=..density..), 
                        color="black", 
                        fill="gray") +
         geom_density(color="black") + 
         facet_wrap(~clarity) + 
         opts(title="Densities") + 
         scale_x_continuous("Price") + 
         scale_y_continuous("Density")

# Using grid to control placement of plots
library(grid) # load grid
grid.newpage() # ready the plotting device
pushViewport(viewport(layout = grid.layout(2, 2))) # set up plot layout, here a 2x2 grid
print(first, vp=viewport(layout.pos.row=1, layout.pos.col = 1)) # plot 'first' in square (1,1)
print(second, vp=viewport(layout.pos.row=2, layout.pos.col = 1)) # plot 'second' in square (2,1)
print(third, vp=viewport(layout.pos.row=1:2, layout.pos.col = 2)) # plot 'third' in rectangle (1-2, 2)
